const database = require('./database');
const translator = require('./translator');
const config = require('../config/config');
const businessHours = require('./businessHours');
const agentNames = require('../config/agent-names.json');
const adminCommands = require('./adminCommands');

/**
 * Get agent language for translation
 */
function getAgentLanguage(phone) {
  // Try to get from agents table
  const agent = database.getAgent(phone);
  if (agent && agent.language) {
    return agent.language;
  }
  
  // Default to team language
  return config.translation.defaultLanguage;
}

/**
 * Translate message for agent
 */
async function translateForAgent(textEN, phone) {
  const agentLang = getAgentLanguage(phone);
  
  // If agent language is English, return directly
  if (agentLang === 'en') {
    return textEN;
  }
  
  try {
    const result = await translator.translate(textEN, agentLang, 'en');
    return result.translatedText;
  } catch (error) {
    console.error(`⚠️ Error translating for agent ${phone}:`, error.message);
    return textEN; // Return in EN as fallback
  }
}

class CommandHandler {
  constructor() {
    // Temporary client states during flows
    this.clientStates = new Map();
    
    // Pending transfers management
    // Map<agentPhone, {ticketId, fromAgent, clientPhone, clientName, reason, timestamp, notifyClient}>
    this.pendingTransfers = new Map();
    
    // Agent busy status
    // Map<agentPhone, {busy: boolean, currentTicketId, waitingTransfers: []}>
    this.agentStatus = new Map();
  }

  /**
   * Check if message is a command
   * @param {string} message - Received message
   * @returns {boolean}
   */
  isCommand(message) {
    if (!message || typeof message !== 'string') {
      return false;
    }
    return message.trim().startsWith('/');
  }

  /**
   * Process command
   * @param {string} message - Message with command
   * @param {string} phone - Client phone number
   * @param {boolean} isAgent - If it's an agent sending
   * @returns {Promise<object>} - {success, response, action}
   */
  async processCommand(message, phone, isAgent = false) {
    const parts = message.trim().split(' ');
    let command = parts[0].toLowerCase();
    let args = parts.slice(1);

    // Normalize compound commands (e.g., "/add agent" -> "/addagent")
    // Skip for multi-word admin commands that need full args
    if (args.length > 0 && 
        !(command === '/add' && args[0] === 'agent' && args[1] === 'to') &&
        !(command === '/remove' && args[0] === 'agent' && args[1] === 'from') &&
        !(command === '/set' && args[0] === 'general') &&
        !(command === '/edit' && args[0] === 'department') &&
        !(command === '/delete' && args[0] === 'department') &&
        !(command === '/create' && args[0] === 'department') &&
        !(command === '/list' && args[0] === 'departments') &&
        !(command === '/toggle' && args[0] === 'departments') &&
        !(command === '/department' && args[0] === 'info')) {
      const compound = `${command}${args[0]}`;
      const validCompounds = [
        '/addagent', '/removeagent', '/editagent', '/listagents', '/agentstats', '/allstats',
        '/changehours', '/changebotname', '/changecompany', '/adminhelp',
        '/listtickets', '/closedtickets', '/closeticket', '/skipticket', '/openticket',
        '/transferticket', '/accepttransfer', '/rejectransfer', '/notifyall', '/setagentnames',
        '/editprofile', '/changelanguage'
      ];
      
      if (validCompounds.includes(compound)) {
        command = compound;
        args = args.slice(1);
      }
    }

    // Check for admin commands first
    if (adminCommands.isAdmin(phone)) {
      const adminResult = await adminCommands.processAdminCommand(phone, command, args);
      if (adminResult) {
        return adminResult;
      }
    }

    // Available commands (multilingual)
    switch (command) {
      // Abrir ticket (pt, en, es, fr, de, it, ru)
      case '/abrir':
      case '/open':
      case '/ouvrir':
      case '/öffnen':     // alemão
      case '/aprire':     // italiano
      case '/открыть':    // russo
        if (args[0] === 'ticket' || !args[0]) {
          return await this.handleOpenTicket(phone);
        }
        break;

      // Perfil (pt, en, es, fr, de, it, ru)
      case '/perfil':
      case '/profile':
      case '/profil':
      case '/profilo':    // italiano
      case '/профиль':    // russo
        return await this.handleViewProfile(phone);

      // Editar perfil (pt, en, es, fr)
      case '/editar':
      case '/edit':
      case '/modifier':   // francês
      case '/bearbeiten': // alemão
      case '/modificare': // italiano
        const profileWords = ['perfil', 'profile', 'profil', 'profilo', 'профиль'];
        if (args[0] && profileWords.includes(args[0])) {
          return await this.handleEditProfile(phone);
        }
        // Se não tem argumento, também edita
        if (!args[0]) {
          return await this.handleEditProfile(phone);
        }
        break;

      // Enviar (pt, en, es, fr, de, it, ru)
      case '/enviar':
      case '/send':
      case '/envoyer':
      case '/senden':     // alemão
      case '/inviare':    // italiano
      case '/отправить':  // russo
        // Verifica se é notificação ou ticket
        const sendState = this.clientStates.get(phone);
        if (sendState && sendState.flow === 'notification') {
          return await this.handleSendNotification(phone);
        } else {
          return await this.handleSendTicket(phone);
        }

      // Alterar idioma (pt, en, es, fr, de, it, zh, ja, ko)
      // Permitido para TODOS (clientes e agentes)
      case '/alterar':
      case '/change':
      case '/cambiar':
      case '/changer':
      case '/ändern':      // alemão
      case '/cambiare':    // italiano
      case '/изменить':    // russo
      case '/更改':        // chinês
      case '/変更':        // japonês
      case '/변경':        // coreano
        // Aceita: idioma, language, langue, idioma (es), sprache, lingua, язык, 语言, 言語, 언어
        const languageWords = ['idioma', 'language', 'langue', 'sprache', 'lingua', 'язык', '语言', '言語', '언어'];
        if (args[0] && languageWords.includes(args[0]) && args[1]) {
          return await this.handleChangeLanguage(phone, args[1]);
        }
        // If only has 1 argument, assume it's the language code directly
        if (args[0] && !languageWords.includes(args[0])) {
          return await this.handleChangeLanguage(phone, args[0]);
        }
        break;

      case '/listar':
      case '/list':
        if (args[0] === 'tickets' && isAgent) {
          return await this.handleListTickets(phone);
        }
        break;

      case '/ticket':
        if (args[0] && isAgent) {
          return await this.handleViewTicket(args[0], phone);
        }
        break;

      case '/atender':
      case '/attend':
        if (args[0] && isAgent) {
          return await this.handleAcceptTicket(args[0], phone);
        }
        break;

      // Fechar ticket (pt, en, es, fr)
      case '/fechar':
      case '/close':
      case '/cerrar':
      case '/fermer':
        const closeWords = ['ticket', 'atendimento', 'servicio', 'service'];
        if (args[0] && closeWords.includes(args[0])) {
          if (isAgent) {
            return await this.handleCloseTicket(phone);
          } else {
            return await this.handleClientCloseTicket(phone);
          }
        }
        // If no argument but is close command
        if (!args[0]) {
          if (isAgent) {
            return await this.handleCloseTicket(phone);
          } else {
            return await this.handleClientCloseTicket(phone);
          }
        }
        break;

      // Skip service (agents only)
      case '/pular':
      case '/skip':
      case '/saltar':
        const skipWords = ['atendimento', 'ticket', 'cliente', 'client', 'servicio'];
        if (isAgent && (args[0] && skipWords.includes(args[0]) || !args[0])) {
          return await this.handleSkipClient(phone);
        }
        break;

      // Compound commands (normalized)
      case '/closeticket':
        if (isAgent) {
          return await this.handleCloseTicket(phone);
        } else {
          return await this.handleClientCloseTicket(phone);
        }
        break;

      case '/openticket':
        return await this.handleOpenTicket(phone);

      case '/listtickets':
        if (isAgent) {
          return await this.handleListTickets(phone);
        }
        break;

      case '/skipticket':
        if (isAgent) {
          return await this.handleSkipClient(phone);
        }
        break;

      case '/editprofile':
        return await this.handleEditProfile(phone);

      case '/changelanguage':
        // Allow both agents and clients to change language
        if (args[0]) {
          return await this.handleChangeLanguage(phone, args[0]);
        }
        break;

      // List available agents (agents only)
      case '/agents':
      case '/agentes':
        if (isAgent) {
          return await this.handleListAgents(phone);
        }
        break;

      // List closed tickets (agents only)
      case '/closed':
        if (args[0] === 'tickets' && isAgent) {
          return await this.handleListClosedTickets(phone);
        }
        break;

      // Notify client (agents only)
      case '/notify':
      case '/notificar':
        if (isAgent && args[0]) {
          return await this.handleNotifyClient(phone, args[0]);
        }
        break;

      // Transfer conversation (agents only)
      case '/transfer':
      case '/transferir':
        if (isAgent && args[0]) {
          return await this.handleTransferTicket(phone, args[0]);
        }
        break;

      // Accept transfer (agents only)
      case '/accept':
      case '/aceitar':
        if (isAgent) {
          return await this.handleAcceptTransfer(phone);
        }
        break;

      // Cancel ongoing action
      case '/cancel':
      case '/cancelar':
        const currentState = this.clientStates.get(phone);
        if (currentState) {
          this.clientStates.delete(phone);
          return {
            success: true,
            response: '❌ Operation cancelled.',
            action: 'cancelled'
          };
        } else {
          return {
            success: false,
            response: '⚠️ No operation in progress to cancel.',
            action: null
          };
        }

      // Ajuda (pt, en, es, fr, de, it, ru)
      case '/ajuda':
      case '/help':
      case '/ayuda':      // espanhol
      case '/aide':       // francês
      case '/hilfe':      // alemão
      case '/aiuto':      // italiano
      case '/помощь':     // russo
        return await this.handleHelp(phone, isAgent);

      default:
        const client = database.getClient(phone);
        const language = client?.language || 'en';
        const errorMsgEN = '❌ Command not recognized. Use /help to see available commands.';
        const translatedError = await translator.translateMessage(errorMsgEN, language);
        
        return {
          success: false,
          response: translatedError,
          action: null
        };
    }

    const client = database.getClient(phone);
    const language = client?.language || 'en';
    const incompleteMsgEN = '❌ Invalid or incomplete command.';
    const translatedIncomplete = await translator.translateMessage(incompleteMsgEN, language);
    
    return {
      success: false,
      response: translatedIncomplete,
      action: null
    };
  }

  /**
   * Start ticket opening flow
   */
  async handleOpenTicket(phone) {
    const client = database.getClient(phone);
    const language = client?.language || 'en';
    
    // Check business hours
    if (!businessHours.isBusinessHours()) {
      const outOfHoursMsgEN = `🕐 *Outside Business Hours*

Our business hours are *${config.businessHours.start} to ${config.businessHours.end}* (${config.businessHours.timezone}).

📅 Business days: ${businessHours.getBusinessDaysText()}

You can open a ticket during our business hours.

⏰ Next available: ${businessHours.getNextBusinessHours()}

Thank you for your understanding! 🙏`;
      
      const translatedMessage = await translator.translateMessage(outOfHoursMsgEN, language);
      
      return {
        success: false,
        response: translatedMessage,
        action: 'out_of_hours'
      };
    }
    
    // Verifica se já tem ticket aberto
    const existingTicket = database.getOpenTicket(phone);
    if (existingTicket) {
      const messageEN = `⚠️ *You already have an open ticket (#${existingTicket.id})*

Please wait for our agent to assist you. To open a new ticket, you need to wait for the current ticket to be closed.

💡 Use /help to see other available commands.`;
      
      const translatedMessage = await translator.translateMessage(messageEN, language);
      
      return {
        success: false,
        response: translatedMessage,
        action: null
      };
    }

    // Verifica se cliente já tem dados salvos
    if (client && client.name) {
      // Cliente já tem dados! Inicia fluxo de descrição
      this.clientStates.set(phone, {
        flow: 'ticket_creation',
        step: 'description',
        data: {
          name: client.name,
          account: client.account || null
        }
      });
      
      const messageEN = `📝 *Describe your problem or request*

You can send:
✅ Text messages
✅ Photos
✅ Videos
✅ Audio 🎤
✅ Documents

When finished, send */send* to finalize the ticket.

💡 To edit your data use: /edit profile`;
      
      const translatedMessage = await translator.translateMessage(messageEN, language);
      
      return {
        success: true,
        response: translatedMessage,
        action: 'start_ticket_flow'
      };
    }

    // Cliente não tem dados, inicia fluxo
    this.clientStates.set(phone, {
      flow: 'ticket_creation',
      step: 'name',
      data: {}
    });

    const messageEN = `🎫 *Ticket Opening*

Please provide your *full name*:`;
    const translatedMessage = await translator.translateMessage(messageEN, language);

    return {
      success: true,
      response: translatedMessage,
      action: 'start_ticket_flow'
    };
  }

  /**
   * Processa fluxo de criação de ticket
   */
  async processTicketFlow(phone, message) {
    const state = this.clientStates.get(phone);
    if (!state || state.flow !== 'ticket_creation') {
      return null;
    }

    const client = database.getClient(phone);
    const language = client?.language || 'en';

    switch (state.step) {
      case 'name':
        state.data.name = message;
        state.step = 'account';
        this.clientStates.set(phone, state);
        
        const accountMsgEN = `Thank you, ${message}! 👤

Do you have an *existing account*?
- If yes, send your *email* or *client ID*
- If no, type *skip*`;
        const translatedAccountMsg = await translator.translateMessage(accountMsgEN, language);
        
        return {
          success: true,
          response: translatedAccountMsg,
          action: 'continue_flow'
        };

      case 'account':
        if (message.toLowerCase() === 'pular') {
          state.data.account = null;
        } else {
          state.data.account = message;
        }
        
        state.step = 'description';
        this.clientStates.set(phone, state);
        
        const descriptionMsgEN = `📝 *Describe your problem or request*

You can send:
✅ Text messages
✅ Photos
✅ Videos
✅ Audio 🎤
✅ Documents
✅ Links

When finished, send */send* to finalize the ticket.`;
        const translatedDescriptionMsg = await translator.translateMessage(descriptionMsgEN, language);
        
        return {
          success: true,
          response: translatedDescriptionMsg,
          action: 'continue_flow'
        };

      case 'description':
        // Armazena mensagem temporariamente
        if (!state.data.messages) {
          state.data.messages = [];
        }
        state.data.messages.push({
          type: 'text',
          content: message,
          timestamp: new Date().toISOString()
        });
        this.clientStates.set(phone, state);
        
        const registeredMsgEN = '✅ Message registered. Continue describing or send */send* when finished.';
        const translatedRegisteredMsg = await translator.translateMessage(registeredMsgEN, language);
        
        return {
          success: true,
          response: translatedRegisteredMsg,
          action: 'collecting_messages'
        };

      case 'department':
        // Process department selection
        const departments = database.listDepartments(true);
        const selection = parseInt(message.trim());
        
        if (isNaN(selection) || selection < 1 || selection > departments.length) {
          const errorEN = `❌ Invalid selection. Please send a number between 1 and ${departments.length}`;
          const error = await translator.translateMessage(errorEN, language);
          
          return {
            success: false,
            response: error
          };
        }

        const department = departments[selection - 1];
        
        // Create ticket with selected department
        return await this.createTicketFromState(phone, department.id);
    }

    return null;
  }

  /**
   * Processa fluxo de fechamento de ticket pelo cliente
   */
  async processCloseTicketFlow(phone, message) {
    const state = this.clientStates.get(phone);
    if (!state || state.flow !== 'close_ticket') {
      return null;
    }

    const client = database.getClient(phone);
    const language = client?.language || 'en';

    if (state.step === 'reason') {
      // Cliente enviou o motivo
      if (message.toLowerCase() === 'cancelar' || message.toLowerCase() === 'cancel') {
        this.clientStates.delete(phone);
        
        const cancelMsgEN = '❌ Closure cancelled. Your ticket remains open.';
        const translatedCancel = await translator.translateMessage(cancelMsgEN, language);
        
        return {
          success: true,
          response: translatedCancel,
          action: 'close_cancelled'
        };
      }

      const ticket = database.getTicket(state.ticketId);
      const reason = message;

      // Fecha o ticket
      database.updateTicket(state.ticketId, {
        status: 'closed',
        closed_by: 'client'
      });

      // Limpa estado
      this.clientStates.delete(phone);

      const successMsgEN = `✅ *Ticket #${state.ticketId} Closed*

📝 Reason: ${reason}

Thank you for using our services!
For new help, use */open ticket*`;

      const translatedSuccess = await translator.translateMessage(successMsgEN, language);

      return {
        success: true,
        response: translatedSuccess,
        action: 'ticket_closed_by_client',
        ticketId: state.ticketId,
        clientPhone: phone,
        clientName: ticket.client_name,
        agentPhone: ticket.agent_number,
        reason: reason,
        language: language
      };
    }

    return null;
  }

  /**
   * Envia ticket
   */
  async handleSendTicket(phone) {
    const state = this.clientStates.get(phone);
    const client = database.getClient(phone);
    const language = client?.language || 'en';
    
    if (!state || state.flow !== 'ticket_creation' || state.step !== 'description') {
      const errorMsgEN = '❌ There is no ticket being created.';
      const translatedError = await translator.translateMessage(errorMsgEN, language);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }

    if (!state.data.messages || state.data.messages.length === 0) {
      const warningMsgEN = '⚠️ You need to send at least one message before finalizing the ticket.';
      const translatedWarning = await translator.translateMessage(warningMsgEN, language);
      
      return {
        success: false,
        response: translatedWarning,
        action: null
      };
    }

    // Check if departments are enabled
    if (database.isDepartmentsEnabled()) {
      const departments = database.listDepartments(true);
      
      if (departments.length > 0) {
        // Move to department selection step
        state.step = 'department';
        this.clientStates.set(phone, state);
        
        let response = `🏢 *Select Department*\n\n`;
        response += `Please choose the department for your request:\n\n`;
        
        departments.forEach((dept, index) => {
          response += `*${index + 1}.* ${dept.name}\n`;
          if (dept.description) {
            response += `   ${dept.description}\n`;
          }
          response += `\n`;
        });
        
        response += `💡 Send the department number (1-${departments.length})`;
        
        const translatedResponse = await translator.translateMessage(response, language);
        
        return {
          success: true,
          response: translatedResponse,
          action: 'awaiting_department_selection'
        };
      }
    }

    // If departments not enabled, create ticket directly
    return await this.createTicketFromState(phone, null);
  }

  async createTicketFromState(phone, departmentId) {
    const state = this.clientStates.get(phone);
    const client = database.getClient(phone);
    const language = client?.language || 'en';
    
    // Cria ticket no banco de dados
    const ticketId = database.createTicket(
      phone,
      state.data.name,
      language,
      departmentId || state.data.department
    );

    // Salva informações do cliente e confirma idioma
    database.saveClient(phone, {
      name: state.data.name,
      email: state.data.account?.includes('@') ? state.data.account : null,
      client_id: state.data.account && !state.data.account.includes('@') ? state.data.account : null,
      language: language,
      language_confirmed: 1  // Confirma idioma após primeiro ticket
    });

    // Update ticket with department if provided
    if (departmentId) {
      const dept = database.getDepartment(departmentId);
      database.updateTicket(ticketId, {
        department_id: departmentId,
        department_name: dept ? dept.name : null
      });
    }

    // Adiciona mensagens ao ticket
    for (const msg of state.data.messages) {
      database.addMessage(
        ticketId,
        phone,
        'client',
        msg.content,     // Corrigido: content primeiro
        msg.type,        // Corrigido: messageType depois
        null,  // translated_content (será traduzido quando agente atender)
        msg.media_data || msg.media_url || null,  // Store base64 data or URL
        language,
        config.translation.defaultLanguage
      );
    }

    // Limpa estado do cliente
    this.clientStates.delete(phone);

    let successMsgEN = `✅ *Ticket #${ticketId} created successfully!*\n\n`;
    successMsgEN += `📋 Name: ${state.data.name}\n`;
    successMsgEN += `🆔 Ticket: #${ticketId}\n`;
    
    if (departmentId) {
      const dept = database.getDepartment(departmentId);
      if (dept) {
        successMsgEN += `🏢 Department: ${dept.name}\n`;
      }
    }
    
    successMsgEN += `📅 Date: ${new Date().toLocaleString('en-US', { timeZone: 'Africa/Maputo' })}\n\n`;
    successMsgEN += `Our agent will contact you soon. Thank you for your patience! 🙏`;
    
    const translatedSuccessMsg = await translator.translateMessage(successMsgEN, language);

    return {
      success: true,
      response: translatedSuccessMsg,
      action: 'ticket_created',
      ticketId: ticketId,
      departmentId: departmentId
    };
  }

  /**
   * Altera idioma do cliente
   */
  async handleChangeLanguage(phone, languageCode) {
    const client = database.getClient(phone);
    const currentLanguage = client?.language || 'en';
    
    // Verifica se há ticket aberto
    const openTicket = database.getOpenTicket(phone);
    if (openTicket) {
      const warningMsgEN = `⚠️ *Cannot change language with open ticket*

You have ticket #${openTicket.id} in progress.

Please wait for the ticket to be closed to change your language.

💡 This ensures the conversation continues in the same language from start to finish.`;
      
      const translatedWarning = await translator.translateMessage(warningMsgEN, currentLanguage);
      
      return {
        success: false,
        response: translatedWarning,
        action: null
      };
    }
    
    // Remove validação de lista fixa - aceita qualquer código
    if (!languageCode || languageCode.length < 2 || languageCode.length > 5) {
      const errorMsgEN = `❌ Invalid language code.

Use codes like: pt, en, es, fr, de, it, zh, ja, ko, ru, ar, hi, etc.

Example: /change language en`;
      
      const translatedError = await translator.translateMessage(errorMsgEN, currentLanguage);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }

    // Atualiza idioma do cliente
    database.saveClient(phone, {
      language: languageCode,
      language_confirmed: 1
    });

    const languageName = translator.getLanguageName(languageCode);
    
    const successMsgEN = `✅ Language changed to *${languageName}*

All future messages will be translated to ${languageName}.`;
    
    const translatedSuccess = await translator.translateMessage(successMsgEN, languageCode);

    return {
      success: true,
      response: translatedSuccess,
      action: 'language_changed',
      language: languageCode
    };
  }

  /**
   * Lista todos os tickets abertos (apenas para agentes)
   */
  async handleListTickets(phone) {
    const tickets = database.listOpenTickets();
    
    if (tickets.length === 0) {
      const messageEN = '📋 *No open tickets at the moment*';
      const message = await translateForAgent(messageEN, phone);
      return {
        success: true,
        response: message,
        action: null
      };
    }

    let responseEN = `📋 *Open Tickets (${tickets.length})*\n\n`;
    
    for (const ticket of tickets) {
      const status = ticket.agent_name ? '🟢 In service' : '🔴 Waiting';
      const agent = ticket.agent_name ? `\n👤 Agent: ${ticket.agent_name}` : '';
      const time = new Date(ticket.created_at).toLocaleString('en-US', { timeZone: 'Africa/Maputo' });
      
      responseEN += `━━━━━━━━━━━━━━━━━━━
🎫 *#${ticket.id}* - ${status}
👤 Client: ${ticket.client_name}
📞 Phone: ${ticket.phone}
🌍 Language: ${translator.getLanguageName(ticket.language)}
📅 Opened: ${time}${agent}

`;
    }

    responseEN += `━━━━━━━━━━━━━━━━━━━
💡 Use */atender or /attend [phone]* to accept a ticket`;

    const response = await translateForAgent(responseEN, phone);

    return {
      success: true,
      response: response,
      action: null,
      tickets: tickets
    };
  }

  /**
   * Visualiza histórico de um ticket
   */
  async handleViewTicket(ticketId, agentPhone) {
    const ticket = database.getTicket(ticketId);
    
    if (!ticket) {
      return {
        success: false,
        response: `❌ Ticket #${ticketId} not found`,
        action: null
      };
    }

    const messages = database.getTicketMessages(ticketId);
    const time = new Date(ticket.created_at).toLocaleString('en-US', { timeZone: 'Africa/Maputo' });
    
    let response = `🎫 *Ticket #${ticket.id}*
━━━━━━━━━━━━━━━━━━━
👤 Client: ${ticket.client_name}
📞 Phone: ${ticket.phone}
🌍 Language: ${translator.getLanguageName(ticket.language)}
📊 Status: ${ticket.status}
📅 Opened: ${time}
${ticket.agent_name ? `👨‍💼 Agent: ${ticket.agent_name}` : ''}

📝 *Message History:*
━━━━━━━━━━━━━━━━━━━\n\n`;

    for (const msg of messages) {
      const msgTime = new Date(msg.created_at).toLocaleTimeString('en-US');
      const sender = msg.sender_type === 'client' ? '👤 Client' : '👨‍💼 Agent';
      
      response += `[${msgTime}] ${sender}:\n${msg.content}\n`;
      if (msg.translated_content && msg.translated_content !== msg.content) {
        response += `🌐 Translated: ${msg.translated_content}\n`;
      }
      response += `\n`;
    }

    return {
      success: true,
      response: response,
      action: null,
      ticket: ticket
    };
  }

  /**
   * Agente aceita atendimento de ticket
   */
  async handleAcceptTicket(clientPhone, agentPhone) {
    const ticket = database.getOpenTicket(clientPhone);
    
    if (!ticket) {
      const errorEN = `❌ No open ticket for number ${clientPhone}`;
      const error = await translateForAgent(errorEN, agentPhone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    if (ticket.agent_name) {
      const errorEN = `⚠️ Ticket #${ticket.id} is already being handled by ${ticket.agent_name}`;
      const error = await translateForAgent(errorEN, agentPhone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    // Get agent info from database
    const agentData = database.getAgent(agentPhone);
    let agentName;
    
    // Check global setting for agent names mode
    const namesMode = database.getSetting('agent_names_mode') || '1'; // Default to real names
    
    if (agentData && namesMode === '1') {
      // Use real name
      agentName = agentData.name;
    } else {
      // Use random name
      agentName = agentNames.names[Math.floor(Math.random() * agentNames.names.length)];
    }
    
    // Update ticket
    database.updateTicket(ticket.id, {
      agent_name: agentName,
      agent_number: agentPhone
    });

    // Increment agent statistics if registered
    if (agentData) {
      database.incrementAgentStats(agentPhone);
    }

    const responseEN = `✅ *Service started*

🎫 Ticket: #${ticket.id}
👤 Client: ${ticket.client_name}
👨‍💼 Agent: ${agentName}

Your messages will be automatically translated to ${translator.getLanguageName(ticket.language)}.`;
    const response = await translateForAgent(responseEN, agentPhone);
    
    return {
      success: true,
      response: response,
      action: 'ticket_accepted',
      ticketId: ticket.id,
      clientPhone: clientPhone,
      agentName: agentName,
      clientLanguage: ticket.language
    };
  }

  /**
   * Fecha ticket
   */
  async handleCloseTicket(agentPhone) {
    // Encontra ticket que o agente está atendendo
    const tickets = database.listOpenTickets();
    const ticket = tickets.find(t => t.agent_number === agentPhone);
    
    if (!ticket) {
      const errorEN = '❌ You are not handling any ticket at the moment';
      const error = await translateForAgent(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    // Fecha ticket
    database.updateTicket(ticket.id, {
      status: 'closed'
    });

    const responseEN = `✅ *Ticket #${ticket.id} closed successfully*

📋 Client: ${ticket.client_name}
📞 Phone: ${ticket.phone}

The client will be notified about the closure.`;
    const response = await translateForAgent(responseEN, agentPhone);
    
    return {
      success: true,
      response: response,
      action: 'ticket_closed',
      ticketId: ticket.id,
      clientPhone: ticket.phone,
      clientLanguage: ticket.language
    };
  }

  /**
   * Cliente fecha o próprio ticket
   */
  async handleClientCloseTicket(phone) {
    const ticket = database.getOpenTicket(phone);
    const client = database.getClient(phone);
    const language = client?.language || 'en';
    
    if (!ticket) {
      const errorMsgEN = '❌ You do not have any open ticket at the moment.';
      const translatedError = await translator.translateMessage(errorMsgEN, language);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }

    // Inicia fluxo de fechamento - pergunta motivo
    this.clientStates.set(phone, {
      flow: 'close_ticket',
      step: 'reason',
      ticketId: ticket.id
    });

    const askReasonEN = `🔐 *Close Ticket #${ticket.id}*

📝 Please tell us the reason for closure:

💡 Examples:
- "Problem solved"
- "I no longer need help"
- "I'll try another solution"

Or type "cancel" to keep the ticket open.`;

    const translatedAskReason = await translator.translateMessage(askReasonEN, language);

    return {
      success: true,
      response: translatedAskReason,
      action: 'awaiting_close_reason'
    };
  }

  /**
   * Pula atendimento atual (agente)
   */
  async handleSkipClient(agentPhone) {
    const tickets = database.listOpenTickets();
    const ticket = tickets.find(t => t.agent_number === agentPhone);
    
    if (!ticket) {
      return {
        success: false,
        response: '❌ You are not handling any ticket at the moment.',
        action: null
      };
    }

    // Marca ticket como "pulado" - remove agente mas não fecha
    database.updateTicket(ticket.id, {
      agent_name: null,
      agent_number: null,
      status: 'waiting' // Status especial para tickets pulados
    });

    const responseEN = `⏭️ *Service Skipped*

📋 Ticket: #${ticket.id}
👤 Client: ${ticket.client_name}

💡 The ticket returned to the queue. If the client responds, you will be notified and can resume service.`;

    const response = await translator.translateSystemMessage(responseEN);

    return {
      success: true,
      response: response,
      action: 'ticket_skipped',
      ticketId: ticket.id,
      clientPhone: ticket.phone
    };
  }

  /**
   * Display help
   */
  async handleHelp(phone, isAgent) {
    if (isAgent) {
      return {
        success: true,
        response: `🤖 *WhatsDesk Pro - Agent Commands*

━━━━━━━━━━━━━━━━━━━━━━━━━━━━
📋 TICKET MANAGEMENT
━━━━━━━━━━━━━━━━━━━━━━━━━━━━

• */listar tickets* or */list tickets*
  → List all open tickets

• */closed tickets*
  → List last 20 closed tickets

• */ticket [ID]* 
  → View ticket history by ID

• */atender [phone]* or */attend [phone]*
  → Accept a ticket from queue

• */fechar ticket* or */close ticket*
  → Close current ticket

• */pular* or */skip*
  → Skip current client (back to queue)

━━━━━━━━━━━━━━━━━━━━━━━━━━━━
🔄 TRANSFER SYSTEM
━━━━━━━━━━━━━━━━━━━━━━━━━━━━

• */agents* or */agentes*
  → List all available agents

• */transfer [phone]* or */transferir [phone]*
  → Transfer current conversation

• */accept* or */aceitar*
  → Accept pending transfer

━━━━━━━━━━━━━━━━━━━━━━━━━━━━
📨 NOTIFICATIONS
━━━━━━━━━━━━━━━━━━━━━━━━━━━━

• */notify [phone]* or */notificar [phone]*
  → Send notification to registered client
  → Can include text + media
  → Finish with */send*
  → 🔒 Only works with registered clients

━━━━━━━━━━━━━━━━━━━━━━━━━━━━
ℹ️ INFORMATION
━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ Messages auto-translated
✅ One agent per ticket
✅ Skipped tickets notify on reply
✅ Transfer includes full history
✅ Notify without opening ticket`,
        action: null
      };
    }

    const client = database.getClient(phone) || {};
    const language = client.language || 'en';

    const helpmessageEN = `🤖 *Available Commands*

🎫 *Tickets:*
• */open ticket* - Open a new ticket
• */send* - Finalize ticket submission
• */close ticket* - Close your current ticket

👤 *Profile:*
• */profile* - View your data
• */edit profile* - Update your data

🌍 *Language:*
• */change language [code]* - Change your language
  Examples: pt, en, es, fr, de, it, zh, ja, ko, ru

❓ *Help:*
• */help* - Display this message`;

    const translatedHelp = await translator.translateMessage(helpmessageEN, language);

    return {
      success: true,
      response: translatedHelp,
      action: null
    };
  }

  /**
   * Show client profile
   */
  async handleViewProfile(phone) {
    const client = database.getClient(phone);
    
    if (!client || !client.name) {
      const language = client?.language || 'en';
      const messageEN = `❌ *Profile not found*

You have not provided your data yet. Use */open ticket* to start.`;
      const translatedMessage = await translator.translateMessage(messageEN, language);
      
      return {
        success: false,
        response: translatedMessage,
        action: null
      };
    }
    
    const language = client.language || 'en';
    const languageName = translator.getLanguageName(language);
    
    const messageEN = `👤 *Your Profile*

📝 Name: ${client.name}
📧 Account: ${client.account || 'Not provided'}
🌍 Language: ${languageName}
📅 Registration: ${client.created_at}

💡 To edit use: /edit profile`;
    
    const translatedMessage = await translator.translateMessage(messageEN, language);
    
    return {
      success: true,
      response: translatedMessage,
      action: null
    };
  }

  /**
   * Start profile editing
   */
  async handleEditProfile(phone) {
    const client = database.getClient(phone);
    const language = client?.language || 'en';
    
    // Inicia fluxo de edição
    this.clientStates.set(phone, {
      flow: 'profile_edit',
      step: 'name',
      data: {}
    });
    
    const messageEN = `✏️ *Profile Editing*

Please provide your *full name*:`;
    const translatedMessage = await translator.translateMessage(messageEN, language);
    
    return {
      success: true,
      response: translatedMessage,
      action: 'start_profile_edit'
    };
  }

  /**
   * Process profile editing flow
   */
  async processProfileEditFlow(phone, message) {
    const state = this.clientStates.get(phone);
    if (!state || state.flow !== 'profile_edit') {
      return null;
    }

    const client = database.getClient(phone);
    const language = client?.language || 'en';

    switch (state.step) {
      case 'name':
        state.data.name = message;
        state.step = 'account';
        this.clientStates.set(phone, state);
        
        const nameMsgEN = `Thank you! 👤

Now send your *account email*
Or type *skip* if you don't have one:`;
        const translatedNameMsg = await translator.translateMessage(nameMsgEN, language);
        
        return {
          success: true,
          response: translatedNameMsg,
          action: 'continue_flow'
        };

      case 'account':
        if (message.toLowerCase() === 'pular' || message.toLowerCase() === 'skip' || message.toLowerCase() === 'saltar' || message.toLowerCase() === 'passer') {
          state.data.account = null;
        } else {
          state.data.account = message;
        }
        
        // Salva dados atualizados no banco
        database.saveClient(phone, {
          name: state.data.name,
          account: state.data.account,
          language: language,
          language_confirmed: 1
        });
        
        this.clientStates.delete(phone);
        
        const successMsgEN = `✅ *Profile updated successfully!*

👤 Name: ${state.data.name}
📧 Account: ${state.data.account || 'Not provided'}

💡 Use /profile to view your data`;
        
        const translatedSuccessMsg = await translator.translateMessage(successMsgEN, language);
        
        return {
          success: true,
          response: translatedSuccessMsg,
          action: 'profile_updated'
        };
    }

    return null;
  }

  /**
   * Lista agentes disponíveis
   */
  async handleListAgents(phone) {
    const adminNumbers = config.admins;
    const agentsList = [];
    
    for (const adminPhone of adminNumbers) {
      const status = this.agentStatus.get(adminPhone) || { busy: false, currentTicketId: null };
      const ticket = status.busy ? database.getTicket(status.currentTicketId) : null;
      
      agentsList.push({
        phone: adminPhone,
        busy: status.busy,
        clientName: ticket ? ticket.client_name : null,
        ticketId: ticket ? ticket.id : null
      });
    }
    
    let messageEN = `👥 *Available Agents*\n\n`;
    
    for (const agent of agentsList) {
      const agentNumber = agent.phone;
      if (agent.busy) {
        messageEN += `📞 ${agentNumber}\n   ❌ Busy - Handling ${agent.clientName} (#${agent.ticketId})\n\n`;
      } else {
        messageEN += `📞 ${agentNumber}\n   ✅ Available\n\n`;
      }
    }
    
    messageEN += `💡 Use: /transfer ${agentsList[0].phone}`;
    
    const translatedMessage = await translator.translateSystemMessage(messageEN);
    
    return {
      success: true,
      response: translatedMessage,
      action: null
    };
  }

  /**
   * Transfer ticket to another agent
   */
  async handleTransferTicket(phone, targetAgentPhone) {
    // Remove non-numeric characters
    targetAgentPhone = targetAgentPhone.replace(/\D/g, '');
    
    // Check if current agent is handling someone
    const ticket = database.getOpenTicketByAgent(phone);
    if (!ticket) {
      const errorMsgEN = '❌ You are not handling any client at the moment.';
      const translatedError = await translator.translateSystemMessage(errorMsgEN);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }
    
    // Check if target agent is valid
    if (!config.admins.includes(targetAgentPhone)) {
      const errorMsgEN = '❌ Invalid agent number.';
      const translatedError = await translator.translateSystemMessage(errorMsgEN);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }
    
    // Check if not trying to transfer to self
    if (targetAgentPhone === phone) {
      const errorMsgEN = '❌ You cannot transfer to yourself.';
      const translatedError = await translator.translateSystemMessage(errorMsgEN);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }
    
    // Start transfer flow (ask for reason)
    this.clientStates.set(phone, {
      flow: 'transfer',
      step: 'reason',
      data: {
        ticketId: ticket.id,
        clientPhone: ticket.phone,
        clientName: ticket.client_name,
        targetAgent: targetAgentPhone
      }
    });
    
    const messageEN = `🔄 *Conversation Transfer*

📋 Ticket: #${ticket.id}
👤 Client: ${ticket.client_name}
📞 To: ${targetAgentPhone}

📝 Please provide the *transfer reason*:

Example:
- "Client needs advanced technical support"
- "Need help with DNS issue"
- "Transferring to billing specialist"

Type "cancel" to cancel transfer.`;
    
    const translatedMessage = await translator.translateSystemMessage(messageEN);
    
    return {
      success: true,
      response: translatedMessage,
      action: 'start_transfer'
    };
  }

  /**
   * Processa fluxo de transferência
   */
  async processTransferFlow(phone, message) {
    const state = this.clientStates.get(phone);
    if (!state || state.flow !== 'transfer') {
      return null;
    }
    
    if (state.step === 'reason') {
      // Cancel transfer
      if (message.toLowerCase() === 'cancelar' || message.toLowerCase() === 'cancel') {
        this.clientStates.delete(phone);
        
        const cancelMsgEN = '❌ Transfer cancelled.';
        const translatedCancel = await translator.translateSystemMessage(cancelMsgEN);
        
        return {
          success: true,
          response: translatedCancel,
          action: 'transfer_cancelled'
        };
      }
      
      const reason = message;
      
      // Clear state
      this.clientStates.delete(phone);
      
      // Ask if want to notify client
      this.clientStates.set(phone, {
        flow: 'transfer_notify',
        step: 'confirm',
        data: {
          ...state.data,
          reason: reason
        }
      });
      
      const messageEN = `✅ Reason registered.

❓ Do you want to notify the client about the transfer?

Type *yes* or *no*:`;
      
      const translatedMessage = await translator.translateSystemMessage(messageEN);
      
      return {
        success: true,
        response: translatedMessage,
        action: 'asking_notify'
      };
    }
    
    return null;
  }

  /**
   * Processa confirmação de notificação de transferência
   */
  async processTransferNotifyFlow(phone, message) {
    const state = this.clientStates.get(phone);
    if (!state || state.flow !== 'transfer_notify') {
      return null;
    }
    
    if (state.step === 'confirm') {
      const notifyClient = message.toLowerCase() === 'sim' || message.toLowerCase() === 'yes' || 
                          message.toLowerCase() === 'sí' || message.toLowerCase() === 'oui';
      
      const targetAgent = state.data.targetAgent;
      const reason = state.data.reason;
      const ticketId = state.data.ticketId;
      const clientPhone = state.data.clientPhone;
      const clientName = state.data.clientName;
      
      // Clear state
      this.clientStates.delete(phone);
      
      // Check if target agent is busy
      const targetStatus = this.agentStatus.get(targetAgent) || { busy: false, currentTicketId: null, waitingTransfers: [] };
      
      if (targetStatus.busy) {
        // Agent busy - add to waiting queue
        targetStatus.waitingTransfers = targetStatus.waitingTransfers || [];
        targetStatus.waitingTransfers.push({
          ticketId,
          fromAgent: phone,
          clientPhone,
          clientName,
          reason,
          notifyClient,
          timestamp: Date.now()
        });
        this.agentStatus.set(targetAgent, targetStatus);
        
        const busyMsgEN = `⚠️ *Agent Busy*

Agent ${targetAgent} is handling another client at the moment.

💡 The transfer will be queued and they will be notified automatically when they finish the current service.`;
        
        const translatedBusy = await translator.translateSystemMessage(busyMsgEN);
        
        return {
          success: true,
          response: translatedBusy,
          action: 'transfer_queued',
          targetAgent,
          ticketId,
          clientPhone,
          fromAgent: phone
        };
      } else {
        // Agent available - create pending transfer
        this.pendingTransfers.set(targetAgent, {
          ticketId,
          fromAgent: phone,
          clientPhone,
          clientName,
          reason,
          notifyClient,
          timestamp: Date.now()
        });
        
        // Update ticket to "transferring" status
        database.updateTicket(ticketId, {
          status: 'transferring'
        });
        
        return {
          success: true,
          response: `✅ Transfer initiated! Waiting for ${targetAgent} to accept.`,
          action: 'transfer_pending',
          targetAgent,
          ticketId,
          clientPhone,
          clientName,
          fromAgent: phone,
          reason,
          notifyClient
        };
      }
    }
    
    return null;
  }

  /**
   * Accept pending transfer
   */
  async handleAcceptTransfer(phone) {
    const transfer = this.pendingTransfers.get(phone);
    
    if (!transfer) {
      const errorMsgEN = '❌ You have no pending transfer.';
      const translatedError = await translator.translateSystemMessage(errorMsgEN);
      
      return {
        success: false,
        response: errorMsgEN,
        action: null
      };
    }
    
    // Check if agent is already handling someone
    const currentTicket = database.getOpenTicketByAgent(phone);
    if (currentTicket) {
      const errorMsgEN = `⚠️ *You are already handling another client*

📋 Current ticket: #${currentTicket.id}
👤 Client: ${currentTicket.client_name}

💡 Finish or skip the current service before accepting transfers.`;
      
      const translatedError = await translator.translateSystemMessage(errorMsgEN);
      
      return {
        success: false,
        response: translatedError,
        action: null
      };
    }
    
    // Remove pending transfer
    this.pendingTransfers.delete(phone);
    
    // Get agent info from database
    const agentData = database.getAgent(phone);
    let newAgentName;
    
    // Check global setting for agent names mode
    const namesMode = database.getSetting('agent_names_mode') || '1'; // Default to real names
    
    if (agentData && namesMode === '1') {
      // Use real name
      newAgentName = agentData.name;
    } else {
      // Use random name
      newAgentName = agentNames.names[Math.floor(Math.random() * agentNames.names.length)];
    }
    
    // Update ticket in database
    database.updateTicket(transfer.ticketId, {
      agent_name: newAgentName,
      agent_number: phone,
      status: 'assigned'
    });

    // Increment agent statistics if registered
    if (agentData) {
      database.incrementAgentStats(phone);
    }
    
    // Update NEW agent status (busy)
    this.agentStatus.set(phone, {
      busy: true,
      currentTicketId: transfer.ticketId,
      waitingTransfers: []
    });
    
    // Free the ORIGINAL agent (no longer handling)
    const fromAgentStatus = this.agentStatus.get(transfer.fromAgent) || { busy: false, currentTicketId: null, waitingTransfers: [] };
    this.agentStatus.set(transfer.fromAgent, {
      busy: false,
      currentTicketId: null,
      waitingTransfers: fromAgentStatus.waitingTransfers || []
    });
    
    // Fetch ticket history
    const messages = database.getTicketMessages(transfer.ticketId);
    
    return {
      success: true,
      response: `✅ *Transfer Accepted!*

📋 Ticket: #${transfer.ticketId}
👤 Client: ${transfer.clientName}
👨‍💼 Agent Name: ${newAgentName}

You are now handling this conversation.`,
      action: 'transfer_accepted',
      transfer: transfer,
      messages: messages,
      newAgentName: newAgentName
    };
  }

  /**
   * Update agent busy status
   */
  updateAgentStatus(agentPhone, busy, ticketId = null) {
    const currentStatus = this.agentStatus.get(agentPhone) || { busy: false, currentTicketId: null, waitingTransfers: [] };
    
    this.agentStatus.set(agentPhone, {
      busy: busy,
      currentTicketId: ticketId,
      waitingTransfers: currentStatus.waitingTransfers || []
    });
    
    // If agent became free and has waiting transfers, process the first one
    if (!busy && currentStatus.waitingTransfers && currentStatus.waitingTransfers.length > 0) {
      const waitingTransfer = currentStatus.waitingTransfers.shift();
      this.agentStatus.set(agentPhone, {
        busy: false,
        currentTicketId: null,
        waitingTransfers: currentStatus.waitingTransfers
      });
      
      // Create pending transfer
      this.pendingTransfers.set(agentPhone, waitingTransfer);
      
      return {
        hasWaitingTransfer: true,
        transfer: waitingTransfer
      };
    }
    
    return { hasWaitingTransfer: false };
  }

  /**
   * Get pending transfers for an agent
   */
  getPendingTransfer(agentPhone) {
    return this.pendingTransfers.get(agentPhone);
  }

  /**
   * List closed tickets
   */
  async handleListClosedTickets(phone) {
    const closedTickets = database.listClosedTickets(20);
    
    if (closedTickets.length === 0) {
      return {
        success: true,
        response: '📋 *Closed Tickets*\n\nNo closed tickets found.',
        action: null
      };
    }
    
    let message = `📋 *Closed Tickets (Last 20)*\n\n`;
    
    for (const ticket of closedTickets) {
      const closedDate = new Date(ticket.closed_at).toLocaleString('en-US');
      message += `━━━━━━━━━━━━━━━━━━━━━━\n`;
      message += `🎫 Ticket #${ticket.id}\n`;
      message += `👤 Client: ${ticket.client_name}\n`;
      message += `📞 Phone: ${ticket.phone}\n`;
      message += `👨‍💼 Agent: ${ticket.agent_name || 'N/A'}\n`;
      message += `📅 Closed: ${closedDate}\n`;
      message += `🌍 Language: ${ticket.language}\n\n`;
    }
    
    message += `💡 Use: /ticket [ID] to view history\n`;
    message += `💡 Use: /notify [phone] to send notification`;
    
    return {
      success: true,
      response: message,
      action: null
    };
  }

  /**
   * Start notification for client
   */
  async handleNotifyClient(agentPhone, clientPhone) {
    // Remove non-numeric characters
    clientPhone = clientPhone.replace(/\D/g, '');
    
    // Check if client exists in database
    const client = database.getClient(clientPhone);
    
    if (!client) {
      return {
        success: false,
        response: `❌ *Security Alert*\n\nClient ${clientPhone} not found in database.\n\n🔒 For security, you can only notify registered clients.`,
        action: null
      };
    }
    
    // Start notification flow
    this.clientStates.set(agentPhone, {
      flow: 'notification',
      step: 'message',
      data: {
        clientPhone: clientPhone,
        clientName: client.name,
        clientLanguage: client.language || 'en',
        messages: []
      }
    });
    
    return {
      success: true,
      response: `📨 *Client Notification*

👤 Client: ${client.name}
📞 Phone: ${clientPhone}
🌍 Language: ${client.language || 'en'}

📝 *Send your notification message*

You can send:
✅ Text messages
✅ Photos
✅ Videos
✅ Audio
✅ Documents

When finished, type */send* to deliver the notification.

💡 Type *cancel* to abort.`,
      action: 'start_notification',
      clientPhone: clientPhone,
      clientName: client.name
    };
  }

  /**
   * Process notification flow
   */
  async processNotificationFlow(agentPhone, message) {
    const state = this.clientStates.get(agentPhone);
    if (!state || state.flow !== 'notification') {
      return null;
    }
    
    if (state.step === 'message') {
      // Cancel notification
      if (message.toLowerCase() === 'cancel' || message.toLowerCase() === 'cancelar') {
        this.clientStates.delete(agentPhone);
        
        return {
          success: true,
          response: '❌ Notification cancelled.',
          action: 'notification_cancelled'
        };
      }
      
      // Add message to array
      state.data.messages.push({
        content: message,
        type: 'text'
      });
      
      return {
        success: true,
        response: `✅ Message added!\n\n📝 Total messages: ${state.data.messages.length}\n\n💡 Send more or type */send* to deliver.`,
        action: 'message_added'
      };
    }
    
    return null;
  }

  /**
   * Add media to notification flow
   */
  addMediaToNotification(agentPhone, mediaData) {
    const state = this.clientStates.get(agentPhone);
    if (!state || state.flow !== 'notification') {
      return false;
    }
    
    state.data.messages.push({
      content: mediaData.caption || `[${mediaData.type}]`,
      type: mediaData.type,
      media: mediaData
    });
    
    return true;
  }

  /**
   * Send notification to client
   */
  async handleSendNotification(agentPhone) {
    const state = this.clientStates.get(agentPhone);
    
    if (!state || state.flow !== 'notification') {
      return {
        success: false,
        response: '❌ No notification in progress.',
        action: null
      };
    }
    
    if (!state.data.messages || state.data.messages.length === 0) {
      return {
        success: false,
        response: '⚠️ You need to send at least one message before delivering the notification.',
        action: null
      };
    }
    
    // Clear state
    this.clientStates.delete(agentPhone);
    
    return {
      success: true,
      response: `✅ *Notification Sent!*

👤 Client: ${state.data.clientName}
📞 Phone: ${state.data.clientPhone}
📨 Messages: ${state.data.messages.length}

The client will receive the notification with context.`,
      action: 'notification_sent',
      clientPhone: state.data.clientPhone,
      clientName: state.data.clientName,
      clientLanguage: state.data.clientLanguage,
      messages: state.data.messages
    };
  }

  /**
   * Get client state
   */
  getClientState(phone) {
    return this.clientStates.get(phone);
  }

  /**
   * Update client state
   */
  updateClientState(phone, state) {
    this.clientStates.set(phone, state);
  }
}

module.exports = new CommandHandler();
